# TgeBrowser Node.js SDK

TgeBrowser API 的 Node.js SDK，支持使用 Selenium 和 Puppeteer 进行浏览器自动化操作。

## 安装依赖

```bash
npm install
```

或使用 yarn:

```bash
yarn install
```

## 依赖包

- **axios**: HTTP 客户端，用于 API 请求
- **selenium-webdriver**: Selenium WebDriver，用于浏览器自动化
- **chromedriver**: Chrome 驱动程序
- **puppeteer**: Puppeteer，用于无头浏览器自动化

## 快速开始

### 1. 基础 API 使用

运行主程序，演示完整的环境生命周期：

```bash
npm start
```

或直接运行：

```bash
node index.js
```

### 2. 使用 Selenium

```bash
node tge_selenium.js
```

示例代码展示如何：
- 启动浏览器环境
- 使用 Selenium 连接到浏览器
- 自动化操作（访问网页、输入文本、点击按钮）
- 关闭浏览器并停止环境

### 3. 使用 Puppeteer

```bash
node tge_puppeteer.js
```

示例代码展示如何：
- 启动浏览器环境
- 通过 CDP 协议连接到浏览器
- 创建新页面并访问网站
- 关闭页面并停止环境

## API 文档

### TgeBrowserAPI 类

#### 构造函数

```javascript
const api = new TgeBrowserAPI(baseUrl, token);
```

参数：
- `baseUrl`: API 服务地址，默认 `http://127.0.0.1:50326`
- `token`: 认证令牌

#### 方法

##### 环境管理

- `createEnv()`: 创建新的浏览器环境
- `updateEnv(envId, newName)`: 更新环境配置
- `startEnv(envId)`: 启动环境
- `stopEnv(envId)`: 停止环境
- `deleteEnv(envId)`: 删除环境
- `deleteEnvCache(envId)`: 删除环境缓存

##### 查询方法

- `listEnvs(current, pageSize)`: 获取环境列表
- `listEnvsOpen(current, pageSize)`: 获取已打开的环境列表
- `listGroups(current, pageSize)`: 获取分组列表
- `listProxies(current, pageSize)`: 获取代理列表

##### 窗口排列

- `sortEnvs(envIds)`: 排列环境窗口
- `customSortEnvs(envIds)`: 自定义排列环境窗口

## 使用示例

### 创建并启动环境

```javascript
const TgeBrowserAPI = require('./TgeBrowserAPI');

const api = new TgeBrowserAPI(
    'http://127.0.0.1:50326',
    'your_token_here'
);

// 创建环境
const createRes = await api.createEnv();
const envId = createRes.data.envId;

// 启动环境
const startRes = await api.startEnv(envId);
console.log('浏览器已启动:', startRes);

// 停止环境
await api.stopEnv(envId);
```

### Selenium 示例

```javascript
const { Builder } = require('selenium-webdriver');
const chrome = require('selenium-webdriver/chrome');

// 启动环境并获取连接信息
const res = await api.startEnv(envId);
const driverPath = res.data.driver;
const port = res.data.port;

// 配置并连接
const options = new chrome.Options();
options.debuggerAddress(`127.0.0.1:${port}`);

const driver = await new Builder()
    .forBrowser('chrome')
    .setChromeOptions(options)
    .setChromeService(new chrome.ServiceBuilder(driverPath))
    .build();

// 执行自动化操作
await driver.get('https://www.baidu.com/');
```

### Puppeteer 示例

```javascript
const puppeteer = require('puppeteer');

// 启动环境并获取 WebSocket 地址
const res = await api.startEnv(envId);
const ws = res.data.ws;

// 连接到浏览器
const browser = await puppeteer.connect({
    browserWSEndpoint: ws,
    defaultViewport: null
});

// 创建新页面
const page = await browser.newPage();
await page.goto('https://baidu.com');
```

## 配置说明

### 环境配置参数

创建环境时可以配置以下参数：

- **browserName**: 浏览器名称
- **proxy**: 代理配置（协议、主机、端口、用户名、密码）
- **fingerprint**: 指纹配置（操作系统、分辨率、语言、时区等）
- **startInfo**: 启动配置（起始页、其他配置）

详细配置请参考 `TgeBrowserAPI.js` 中的 `createEnv()` 方法。

## 注意事项

1. 确保 TgeBrowser 服务正在运行
2. 替换示例代码中的 token 为您自己的认证令牌
3. 替换 envId 为实际的环境 ID
4. Selenium 需要正确配置 chromedriver 路径
5. 使用完毕后记得停止环境以释放资源

## 许可证

MIT
